<?php
function panel_base_dir(): string {
  return realpath(__DIR__);
}

function panel_data_dir(int $appId): string {
  $dir = panel_base_dir() . DIRECTORY_SEPARATOR . "data" . DIRECTORY_SEPARATOR . ("app_" . $appId);
  if (!is_dir($dir)) { @mkdir($dir, 0755, true); }
  return $dir;
}

function panel_uploads_dir(int $appId): string {
  $dir = panel_base_dir() . DIRECTORY_SEPARATOR . "uploads" . DIRECTORY_SEPARATOR . ("app_" . $appId);
  if (!is_dir($dir)) { @mkdir($dir, 0755, true); }
  return $dir;
}

function panel_read_json(int $appId, string $file, $default) {
  $dir = panel_data_dir($appId);
  $path = $dir . DIRECTORY_SEPARATOR . $file;
  if (!file_exists($path)) return $default;
  $raw = file_get_contents($path);
  $data = json_decode($raw, true);
  return is_null($data) ? $default : $data;
}

function panel_write_json(int $appId, string $file, $data): bool {
  $dir = panel_data_dir($appId);
  $path = $dir . DIRECTORY_SEPARATOR . $file;
  $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
  return file_put_contents($path, $json) !== false;
}

function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function panel_current_app_id(): int {
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
  $id = intval($_SESSION["app_id"] ?? 4);
  return $id > 0 ? $id : 4;
}

function panel_set_app_id(int $id) {
  if (session_status() !== PHP_SESSION_ACTIVE) session_start();
  $_SESSION["app_id"] = $id;
}

function panel_apps_list(): array {
  $path = panel_base_dir() . "/apps.json";
  if (!file_exists($path)) {
    $default = [
      ["appId"=>4, "name"=>"La Caravana radio show"]
    ];
    file_put_contents($path, json_encode($default, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
    return $default;
  }
  $data = json_decode(file_get_contents($path), true);
  return is_array($data) ? $data : [];
}

function panel_find_app_name(int $appId): string {
  foreach (panel_apps_list() as $a) {
    if (intval($a["appId"] ?? 0) === $appId) return strval($a["name"] ?? ("App " . $appId));
  }
  return "App " . $appId;
}

function panel_base_url(): string {
  $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
  $scheme = $https ? "https" : "http";
  $host = $_SERVER['HTTP_HOST'] ?? "localhost";
  return $scheme . "://" . $host;
}

function panel_save_uploaded_image(int $appId, string $fileField): ?string {
  if (!isset($_FILES[$fileField]) || $_FILES[$fileField]["error"] !== UPLOAD_ERR_OK) return null;

  $tmp = $_FILES[$fileField]["tmp_name"];
  $name = $_FILES[$fileField]["name"] ?? "image";
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  if (!in_array($ext, ["jpg","jpeg","png","webp","gif"])) return null;

  $safe = preg_replace('/[^a-zA-Z0-9_-]+/', '_', pathinfo($name, PATHINFO_FILENAME));
  $filename = $safe . "_" . date("Ymd_His") . "_" . substr(bin2hex(random_bytes(4)),0,8) . "." . $ext;

  $dir = panel_uploads_dir($appId);
  $dest = $dir . DIRECTORY_SEPARATOR . $filename;

  if (!move_uploaded_file($tmp, $dest)) return null;

  // Return absolute URL
  return panel_base_url() . "/panel/uploads/app_" . $appId . "/" . $filename;
}


function panel_save_uploaded_mp3(int $appId, string $fileField): ?string {
  if (!isset($_FILES[$fileField]) || $_FILES[$fileField]["error"] !== UPLOAD_ERR_OK) return null;
  $tmp = $_FILES[$fileField]["tmp_name"];
  $name = $_FILES[$fileField]["name"] ?? "audio.mp3";
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  if ($ext !== "mp3") return null;

  $safe = preg_replace('/[^a-zA-Z0-9_-]+/', '_', pathinfo($name, PATHINFO_FILENAME));
  $filename = $safe . "_" . date("Ymd_His") . "_" . substr(bin2hex(random_bytes(4)),0,8) . ".mp3";

  $dir = panel_uploads_dir($appId);
  $dest = $dir . DIRECTORY_SEPARATOR . $filename;
  if (!move_uploaded_file($tmp, $dest)) return null;

  return panel_base_url() . "/panel/uploads/app_" . $appId . "/" . $filename;
}

// Creates a vertical 9:16 image (1080x1920) using center-crop.
// If GD is not available, it falls back to normal upload.
function panel_save_uploaded_vertical_image(int $appId, string $fileField, int $targetW=1080, int $targetH=1920): ?string {
  if (!isset($_FILES[$fileField]) || $_FILES[$fileField]["error"] !== UPLOAD_ERR_OK) return null;

  $tmp = $_FILES[$fileField]["tmp_name"];
  $name = $_FILES[$fileField]["name"] ?? "image";
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  if (!in_array($ext, ["jpg","jpeg","png","webp","gif"])) return null;

  $safe = preg_replace('/[^a-zA-Z0-9_-]+/', '_', pathinfo($name, PATHINFO_FILENAME));
  $filename = $safe . "_" . date("Ymd_His") . "_" . substr(bin2hex(random_bytes(4)),0,8) . ".jpg"; // normalize to jpg

  $dir = panel_uploads_dir($appId);
  $dest = $dir . DIRECTORY_SEPARATOR . $filename;

  // If GD isn't available, fall back
  if (!function_exists("imagecreatefromjpeg")) {
    $fallback = panel_save_uploaded_image($appId, $fileField);
    return $fallback;
  }

  $src = null;
  if ($ext === "jpg" || $ext === "jpeg") $src = @imagecreatefromjpeg($tmp);
  else if ($ext === "png") $src = @imagecreatefrompng($tmp);
  else if ($ext === "webp" && function_exists("imagecreatefromwebp")) $src = @imagecreatefromwebp($tmp);
  else if ($ext === "gif") $src = @imagecreatefromgif($tmp);

  if (!$src) {
    $fallback = panel_save_uploaded_image($appId, $fileField);
    return $fallback;
  }

  $sw = imagesx($src); $sh = imagesy($src);
  if ($sw <= 0 || $sh <= 0) { imagedestroy($src); return null; }

  // center-crop to 9:16
  $targetRatio = $targetW / $targetH;
  $srcRatio = $sw / $sh;

  if ($srcRatio > $targetRatio) {
    // too wide
    $newW = intval($sh * $targetRatio);
    $newH = $sh;
    $sx = intval(($sw - $newW) / 2);
    $sy = 0;
  } else {
    // too tall
    $newW = $sw;
    $newH = intval($sw / $targetRatio);
    $sx = 0;
    $sy = intval(($sh - $newH) / 2);
  }

  $dst = imagecreatetruecolor($targetW, $targetH);
  // Fill black (for safety)
  $black = imagecolorallocate($dst, 0, 0, 0);
  imagefill($dst, 0, 0, $black);

  imagecopyresampled($dst, $src, 0, 0, $sx, $sy, $targetW, $targetH, $newW, $newH);

  // Save as jpg
  imagejpeg($dst, $dest, 88);
  imagedestroy($src);
  imagedestroy($dst);

  return panel_base_url() . "/panel/uploads/app_" . $appId . "/" . $filename;
}

function panel_http_get_json(string $url, int $timeoutSeconds=6): ?array {
  $ctx = stream_context_create([
    "http" => [
      "timeout" => $timeoutSeconds,
      "user_agent" => "panel/1.0",
    ]
  ]);
  $raw = @file_get_contents($url, false, $ctx);
  if ($raw === false) return null;
  $data = json_decode($raw, true);
  if (!is_array($data)) return null;
  return $data;
}

// Fetches YouTube metadata (title + thumbnail) using oEmbed (no API key required).
function panel_youtube_metadata(string $youtubeId): array {
  $id = trim($youtubeId);
  if ($id === "") return ["title"=>"", "thumbnail"=>""];

  $oembed = "https://www.youtube.com/oembed?format=json&url=" . rawurlencode("https://www.youtube.com/watch?v=".$id);
  $data = panel_http_get_json($oembed);
  if (!$data) return ["title"=>"", "thumbnail"=>""];
  $title = trim($data["title"] ?? "");
  $thumb = trim($data["thumbnail_url"] ?? "");
  return ["title"=>$title, "thumbnail"=>$thumb];
}


function panel_extract_youtube_id(string $input): string {
  $s = trim($input);
  if ($s === "") return "";

  // If looks like an ID already
  if (preg_match('/^[a-zA-Z0-9_-]{11}$/', $s)) return $s;

  // youtu.be/<id>
  if (preg_match('~youtu\.be/([a-zA-Z0-9_-]{11})~', $s, $m)) return $m[1];

  // youtube.com/watch?v=<id>
  if (preg_match('~v=([a-zA-Z0-9_-]{11})~', $s, $m)) return $m[1];

  // youtube.com/embed/<id>
  if (preg_match('~/embed/([a-zA-Z0-9_-]{11})~', $s, $m)) return $m[1];

  return "";
}
?>